function circle = makeCartCircle(radius, num_points, center_pos, arc_angle, plot_circle)
%MAKECARTCIRCLE     Create a 2D Cartesian circle or arc.
%
% DESCRIPTION:
%       MakeCartCircle creates a 2 x num_points array of the Cartesian
%       coordinates of points evenly distributed over a circle or arc (if
%       arc_angle is given).
%
% USAGE:
%       circle = makeCartCircle(radius, num_points)
%       circle = makeCartCircle(radius, num_points, center_pos)
%       circle = makeCartCircle(radius, num_points, center_pos, arc_angle)
%       circle = makeCartCircle(radius, num_points, center_pos, arc_angle, plot_circle)
%
% INPUTS:
%       radius          - circle radius [m]
%       num_points      - number of points in the circle
%
% OPTIONAL INPUTS:
%       center_pos      - [x, z] position of the circle center [m] 
%                         (default = [0, 0])
%       arc_angle       - arc angle for incomplete circle [radians]
%                         (default = 2*pi)
%       plot_circle     - Boolean controlling whether the Cartesian points
%                         are plotted (default = false)
%
% OUTPUTS:
%       circle          - 2 x num_points array of Cartesian coordinates
%
% ABOUT:
%       author          - Bradley Treeby
%       date            - 5th June 2009
%       last update     - 17th July 2009
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
%
% See also cart2grid, makeCartSphere, makeCircle

% check for plot_circle input
if nargin < 5
    plot_circle = false;
end

% check for arc_angle input
if nargin < 4
    arc_angle = 2*pi;
    full_circle = true;
elseif arc_angle == 2*pi;
    full_circle = true;
else
    full_circle = false;
end

% check for center_pos input
if nargin < 3
    cx = 0;
    cz = 0;
else
    cx = center_pos(1);
    cz = center_pos(2);
end

% ensure there is only a total of num_points including the endpoints when
% arc_angle is not equal to 2*pi
if ~full_circle
    num_points = num_points - 1;
end

% create angles
angles = (0:(num_points))*arc_angle/(num_points) + pi/2;

% discard repeated final point if arc_angle is equal to 2*pi
if full_circle
    angles = angles(1:end-1);
end

% create cartesian grid
circle = flipud([radius*cos(angles); radius*sin(-angles)]);

% offset if needed
circle(1, :) = circle(1, :) + cx;
circle(2, :) = circle(2, :) + cz;

% plot results
if plot_circle
    
    % select suitable axis scaling factor
    [x_sc scale prefix] = scaleSI(max(abs(circle(:)))); 
    
    % create the figure
    figure;
    plot(circle(1,:)*scale, circle(2,:)*scale, 'b.');
    set(gca, 'YDir', 'reverse');
    xlabel(['x-position [' prefix 'm]']);
    ylabel(['z-position [' prefix 'm]']);
    axis equal;
    
end